<?php
use WowDiviCarousel\Traits\InstgramFeed;

class WDC_InstagramFeed extends WDC_Builder_Module {

	use InstgramFeed;

	private static $api_url = 'https://graph.instagram.com/';

	private $access_token = null;

	protected $module_credits = array(
		'module_uri' => 'https://wowcarousel.com/modules/instagram-feed/',
		'author'     => 'Wow Carousel',
		'author_uri' => 'https://divipeople.com/wow-divi-carousel/',
	);

	public function init() {

		$this->vb_support             = 'on';
		$this->slug                   = 'wdc_instagram_feed';
		$this->name                   = esc_html__( 'Wow Instagram Feed', 'wdc-wow-divi-carousel' );
		$this->icon_path              = plugin_dir_path( __FILE__ ) . 'instagram-feed.svg';
		$this->settings_modal_toggles = array(

			'general'  => array(
				'toggles' => array(
					'content'            => esc_html__( 'Instagram Feed', 'wdc-wow-divi-carousel' ),
					'instagram_settings' => esc_html__( 'Instagram Settings', 'wdc-wow-divi-carousel' ),
					'settings'           => array(
						'title'             => esc_html__( 'Carousel Settings', 'wdc-wow-divi-carousel' ),
						'tabbed_subtoggles' => true,
						'sub_toggles'       => array(
							'general'  => array(
								'name' => esc_html__( 'General', 'wdc-wow-divi-carousel' ),
							),
							'advanced' => array(
								'name' => esc_html__( 'Advanced', 'wdc-wow-divi-carousel' ),
							),
						),
					),
				),
			),

			'advanced' => array(
				'toggles' => array(
					'instagram'  => esc_html__( 'Instagram', 'wdc-wow-divi-carousel' ),
					'overlay'    => esc_html__( 'Overlay', 'wdc-wow-divi-carousel' ),
					'caption'    => esc_html__( 'Caption', 'wdc-wow-divi-carousel' ),
					'arrow'      => array(
						'title'             => esc_html__( 'Navigation', 'wdc-wow-divi-carousel' ),
						'tabbed_subtoggles' => true,
						'sub_toggles'       => array(
							'arrow_common' => array(
								'name' => esc_html__( 'Common', 'wdc-wow-divi-carousel' ),
							),
							'arrow_left'   => array(
								'name' => esc_html__( 'Prev', 'wdc-wow-divi-carousel' ),
							),
							'arrow_right'  => array(
								'name' => esc_html__( 'Next', 'wdc-wow-divi-carousel' ),
							),
						),
					),
					'pagination' => array(
						'title'             => esc_html__( 'Pagination', 'wdc-wow-divi-carousel' ),
						'tabbed_subtoggles' => true,
						'sub_toggles'       => array(
							'pagi_common' => array(
								'name' => esc_html__( 'Common', 'wdc-wow-divi-carousel' ),
							),
							'pagi_active' => array(
								'name' => esc_html__( 'Active', 'wdc-wow-divi-carousel' ),
							),
						),
					),
				),
			),

		);
	}

	public function get_advanced_fields_config() {

		$advanced_fields                = array();
		$advanced_fields['text']        = false;
		$advanced_fields['text_shadow'] = false;
		$advanced_fields['fonts']       = false;

		$advanced_fields['fonts']['overlay_caption'] = array(
			'label'          => esc_html__( 'Caption', 'wdc-wow-divi-carousel' ),
			'css'            => array(
				'main'      => '%%order_class%% .wdc-insta-overlay .wdc-insta-caption',
				'important' => 'all',
			),
			'range_settings' => array(
				'min'  => '0',
				'max'  => '3',
				'step' => '0.1',
			),
			'toggle_slug'    => 'overlay',
			'tab_slug'       => 'advanced',
		);

		$advanced_fields['fonts']['caption'] = array(
			'css'            => array(
				'main'      => '%%order_class%% .wdc-insta-content .wdc-insta-caption',
				'important' => 'all',
			),
			'range_settings' => array(
				'min'  => '0',
				'max'  => '3',
				'step' => '0.1',
			),
			'toggle_slug'    => 'caption',
			'tab_slug'       => 'advanced',
		);

		return $advanced_fields;

	}

	public function get_fields() {

		$insta_token_text = esc_html__( 'Get Access Token', 'wdc-wow-divi-carousel' );
		$insta_desc_text  = esc_html__( "Authenticated requests need Instagram Access token. The token provides a secure way for a website to ask Instagram's permission to access your profile and display its images.", 'wdc-wow-divi-carousel' );

		$fields = array();

		$fields['endpoint'] = array(
			'label'            => esc_html__( 'What to display', 'wdc-wow-divi-carousel' ),
			'type'             => 'select',
			'default'          => 'self',
			'options'          => array(
				'self' => esc_html__( 'My Photos', 'wdc-wow-divi-carousel' ),
			),
			'computed_affects' => array( '__feed' ),
			'toggle_slug'      => 'content',
		);

		$fields['access_token_type'] = array(
			'label'            => esc_html__( 'Access Token Type', 'wdc-wow-divi-carousel' ),
			'type'             => 'select',
			'default'          => 'custom',
			'options'          => array(
				'custom' => esc_html__( 'Custom', 'wdc-wow-divi-carousel' ),
			),
			'computed_affects' => array( '__feed' ),
			'toggle_slug'      => 'content',
		);

		$fields['access_token'] = array(
			'label'            => esc_html__( 'Access Token', 'wdc-wow-divi-carousel' ),
			'type'             => 'text',
			'description'      => sprintf( '%1$s <a target="_blank" href="https://api.divipeople.com/instagram-token/">%2$s</a>', $insta_desc_text, $insta_token_text ),
			'show_if'          => array(
				'access_token_type' => 'custom',
			),
			'computed_affects' => array( '__feed' ),
			'toggle_slug'      => 'content',
		);

		$fields['cache_timeout'] = array(
			'label'             => esc_html__( 'Data Cache Time', 'wdc-wow-divi-carousel' ),
			'description'       => esc_html__( 'Default value is 1. Set 0 to remove cache immediately.', 'wdc-wow-divi-carousel' ),
			'default'           => '1',
			'default_on_front'  => '1',
			'value_min'         => 0,
			'value_max'         => 100,
			'type'              => 'text',
			'value_type'        => 'float',
			'number_validation' => true,
			'computed_affects'  => array( '__feed' ),
			'toggle_slug'       => 'instagram_settings',
		);

		$fields['cache_timeout_type'] = array(
			'label'            => esc_html__( 'Cache Timeout Type', 'wdc-wow-divi-carousel' ),
			'type'             => 'select',
			'default'          => 'hour',
			'options'          => array(
				'none'   => esc_html__( 'None', 'wdc-wow-divi-carousel' ),
				'minute' => esc_html__( 'Minute', 'wdc-wow-divi-carousel' ),
				'hour'   => esc_html__( 'Hour', 'wdc-wow-divi-carousel' ),
				'day'    => esc_html__( 'Day', 'wdc-wow-divi-carousel' ),
				'week'   => esc_html__( 'Week', 'wdc-wow-divi-carousel' ),
			),
			'computed_affects' => array( '__feed' ),
			'toggle_slug'      => 'instagram_settings',
		);

		$fields['show_on_hover'] = array(
			'label'            => esc_html__( 'Show on hover', 'wdc-wow-divi-carousel' ),
			'type'             => 'multiple_checkboxes',
			'default'          => 'on|off',
			'options'          => array(
				'icon'    => esc_html__( 'Icon', 'wdc-wow-divi-carousel' ),
				'caption' => esc_html__( 'Caption', 'wdc-wow-divi-carousel' ),
			),
			'computed_affects' => array( '__feed' ),
			'toggle_slug'      => 'instagram_settings',
		);

		$fields['posts_counter'] = array(
			'label'            => esc_html__( 'Number of instagram posts', 'wdc-wow-divi-carousel' ),
			'type'             => 'range',
			'default'          => 6,
			'unitless'         => true,
			'range_settings'   => array(
				'min'  => 1,
				'max'  => 100,
				'step' => 1,
			),
			'toggle_slug'      => 'instagram_settings',
			'computed_affects' => array(
				'__feed',
			),
		);

		$fields['enable_image_link'] = array(
			'label'            => __( 'Enable Image Link', 'wdc-wow-divi-carousel' ),
			'type'             => 'yes_no_button',
			'default'          => 'on',
			'options'          => array(
				'on'  => esc_html__( 'Yes', 'wdc-wow-divi-carousel' ),
				'off' => esc_html__( 'No', 'wdc-wow-divi-carousel' ),
			),
			'computed_affects' => array( '__feed' ),
			'toggle_slug'      => 'instagram_settings',
		);

		$fields['show_caption'] = array(
			'label'            => __( 'Show Caption', 'wdc-wow-divi-carousel' ),
			'type'             => 'yes_no_button',
			'default'          => 'off',
			'options'          => array(
				'on'  => esc_html__( 'Yes', 'wdc-wow-divi-carousel' ),
				'off' => esc_html__( 'No', 'wdc-wow-divi-carousel' ),
			),
			'computed_affects' => array( '__feed' ),
			'toggle_slug'      => 'instagram_settings',
		);

		$fields['svg_color'] = array(
			'label'        => esc_html__( 'Instagram Icon Color', 'wdc-wow-divi-carousel' ),
			'type'         => 'color-alpha',
			'default'      => '#000',
			'custom_color' => true,
			'toggle_slug'  => 'overlay',
			'tab_slug'     => 'advanced',
		);

		$fields['insta_icon_size'] = array(
			'label'          => esc_html__( 'Instagram Icon Size', 'wdc-wow-divi-carousel' ),
			'type'           => 'range',
			'default'        => '32px',
			'default_unit'   => 'px',
			'mobile_options' => true,
			'range_settings' => array(
				'min'  => 0,
				'max'  => 72,
				'step' => 1,
			),
			'toggle_slug'    => 'overlay',
			'tab_slug'       => 'advanced',
		);

		$fields['content_padding'] = array(
			'label'          => esc_html__( 'Padding', 'wdc-wow-divi-carousel' ),
			'type'           => 'custom_padding',
			'default'        => '0px|0px|0px|0px',
			'mobile_options' => true,
			'toggle_slug'    => 'caption',
			'tab_slug'       => 'advanced',
		);

		$fields['caption_bg'] = array(
			'label'        => esc_html__( 'Caption Background', 'wdc-wow-divi-carousel' ),
			'type'         => 'color-alpha',
			'custom_color' => true,
			'toggle_slug'  => 'caption',
			'tab_slug'     => 'advanced',
		);

		$fields['image_height'] = array(
			'label'          => esc_html__( 'Image Height', 'wdc-wow-divi-carousel' ),
			'type'           => 'range',
			'default_unit'   => 'px',
			'default'        => 'auto',
			'mobile_options' => true,
			'range_settings' => array(
				'min'  => 0,
				'step' => 1,
				'max'  => 500,
			),
			'toggle_slug'    => 'instagram',
			'tab_slug'       => 'advanced',
		);

		$fields['__feed'] = array(
			'type'                => 'computed',
			'computed_callback'   => array( 'WDC_InstagramFeed', 'render_feed' ),
			'computed_depends_on' => array(
				'endpoint',
				'access_token_type',
				'access_token',
				'cache_timeout',
				'cache_timeout_type',
				'enable_image_link',
				'show_caption',
				'posts_counter',
				'show_on_hover',
				'highlight_every',
			),
		);

		// Overlay.
		$overlay = $this->custom_advanced_background_fields(
			'overlay',
			esc_html__( 'Hover Overlay', 'wdc-wow-divi-carousel' ),
			'advanced',
			'overlay',
			array( 'color', 'gradient', 'image' ),
			array(),
			'rgba(0,0,0,0.25)'
		);

		$carousel_options = WDC_Builder_Module::get_carousel_option_fields( 'carousel', array( 'lightbox' ) );

		return array_merge( $fields, $overlay, $carousel_options );
	}

	public static function render_feed( $args ) {
		$defaults = array(
			'endpoint'           => '',
			'access_token_type'  => '',
			'access_token'       => '',
			'cache_timeout'      => '',
			'cache_timeout_type' => '',
			'enable_image_link'  => '',
			'show_caption'       => '',
			'posts_counter'      => '',
			'show_on_hover'      => '',
			'svg_color'          => '',
			'highlight_every'    => '',
		);

		$args = wp_parse_args( $args, $defaults );

		$html     = '';
		$messages = array();

		if ( 'self' === $args['endpoint'] ) {
			if ( 'custom' === $args['access_token_type'] && empty( $args['access_token'] ) ) {

				$messages['invalid_token_id'] = esc_html__( 'Please, enter Access Token.', 'wdc-wow-divi-carousel' );
			} elseif ( 'default' === $args['access_token_type'] ) {

				$messages['invalid_token_id'] = esc_html__( 'Please, enter Access Token.', 'wdc-wow-divi-carousel' );
			}
		}

		$cache_timeout      = 0 === $args['cache_timeout'] || ! empty( $args['cache_timeout'] ) ? $args['cache_timeout'] : 1;
		$cache_timeout_type = self::cache_timeout_type( $args['cache_timeout_type'] );

		$access_token = $args['access_token'];

		$order_class = self::get_module_order_class( 'wdc_instagram_feed' );
		$module_id   = str_replace( '_', '', str_replace( 'wdc_instagram_feed', '', $order_class ) );

		$transient_key = 'wdc_insta_feed_' . $module_id . str_replace( '.', '_', $access_token );
		$transient_key = substr( $transient_key, 0, 170 );

		if ( '0' === $cache_timeout ) {
			delete_transient( $transient_key );
			$cache_timeout      = 1;
			$cache_timeout_type = MINUTE_IN_SECONDS;
		}

		$data = get_transient( $transient_key );

		if ( $access_token && false === $data ) {

			$url = add_query_arg(
				array(
					'fields'       => 'caption,id,media_type,media_url,permalink,thumbnail_url,timestamp,username',
					'limit'        => 100,
					'access_token' => $access_token,
				),
				self::$api_url . 'me/media/'
			);

			$data = wp_remote_retrieve_body( wp_remote_get( $url ) );
			set_transient( $transient_key, $data, $cache_timeout * $cache_timeout_type );

		}

		$body = json_decode( $data, true );

		if ( empty( $access_token ) || empty( $body ) ) {
			$messages['invalid_token_id'] = __( 'Please, enter Access Token.', 'wdc-wow-divi-carousel' );
		} elseif ( array_key_exists( 'error', $body ) ) {
			$messages['invalid_token'] = $body['error']['message'];
		} elseif ( empty( $body['data'] ) ) {
			$messages['data_empty'] = __( 'Whoops! It seems like this account has not created any post yet.', 'wdc-wow-divi-carousel' );
		}

		if ( ! empty( $messages ) ) {
			foreach ( $messages as $key => $message ) {
				$html = sprintf( '<span class="wdc-insta-error-message">%1$s</span>', esc_html( $message ) );
				return $html;
			}
		}

		$total_data = count( $body['data'] );

		$nodes = array_slice( $body['data'], 0, $args['posts_counter'], true );

		ob_start();

		foreach ( $nodes as $node ) {

			$media_url = ! empty( $node['media_url'] ) ? $node['media_url'] : '';

			switch ( $node['media_type'] ) {
				case 'VIDEO':
					$media_url = ! empty( $node['thumbnail_url'] ) ? $node['thumbnail_url'] : '';
					break;

				case 'CAROUSEL_ALBUM':
					$media_url = ! empty( $node['children']['data'][0]['media_url'] ) ? $node['children']['data'][0]['media_url'] : $media_url;
					break;
			}

			$link      = $node['permalink'];
			$image     = $media_url;
			$caption   = ! empty( $node['caption'] ) ? $node['caption'] : '';
			$comments  = ! empty( $node['comments_count'] ) ? $node['comments_count'] : 0;
			$likes     = ! empty( $node['like_count'] ) ? $node['like_count'] : 0;
			$highlight = false;
			self::ig_feed( $args, $link, $image, $caption, $comments, $likes, $highlight );
		}

		$html = ob_get_clean();

		return $html;
	}

	protected static function cache_timeout_type( $type ) {
		switch ( $type ) {
			case 'minute':
				$cache_time_type = MINUTE_IN_SECONDS;
				break;
			case 'hour':
				$cache_time_type = HOUR_IN_SECONDS;
				break;
			case 'day':
				$cache_time_type = DAY_IN_SECONDS;
				break;
			case 'week':
				$cache_time_type = WEEK_IN_SECONDS;
				break;
			case 'month':
				$cache_time_type = MONTH_IN_SECONDS;
				break;
			case 'year':
				$cache_time_type = YEAR_IN_SECONDS;
				break;
			default:
				$cache_time_type = DAY_IN_SECONDS;
		}
		return $cache_time_type;
	}

	public function render( $attrs, $content, $render_slug ) {

		$this->apply_css( $render_slug );
		$this->apply_carousel_css( $render_slug );

		$sliding_dir      = $this->props['sliding_dir'];
		$is_center        = $this->props['is_center'];
		$center_mode_type = $this->props['center_mode_type'];
		$custom_cursor    = $this->props['custom_cursor'];
		$classes          = array();
		$order_class      = self::get_module_order_class( $render_slug );
		$order_number     = str_replace( '_', '', str_replace( $this->slug, '', $order_class ) );

		if ( 'on' === $is_center ) {
			array_push( $classes, 'wdc-centered' );
			array_push( $classes, "wdc-centered--{$center_mode_type}" );
		}

		if ( 'on' === $custom_cursor ) {
			array_push( $classes, 'wdc-cursor' );
		}

		return sprintf(
			'<div class="wdc-carousel-parent" %2$s dir="%5$s">
                <div class="wdc-carousel wdc-carousel-%4$s wdc-carousel-jq wdc-post-carousel wdc-carousel-frontend %3$s">
                    %1$s
                </div>
            </div>',
			self::render_feed( $this->props ),
			$this->get_carousel_options_data( $order_number, 'post' ),
			join( ' ', $classes ),
			$order_number,
			$sliding_dir
		);
	}

	protected function apply_css( $render_slug ) {

		$this->get_responsive_styles(
			'content_padding',
			'%%order_class%% .wdc-insta-content',
			array( 'primary' => 'padding' ),
			array( 'default' => '0px|0px|0px|0px' ),
			$render_slug
		);

		$this->get_responsive_styles(
			'caption_bg',
			'%%order_class%% .wdc-insta-content',
			array( 'primary' => 'background-color' ),
			array( 'default' => '' ),
			$render_slug
		);

		$this->get_responsive_styles(
			'svg_color',
			'%%order_class%% .wdc-insta-icon',
			array( 'primary' => 'color' ),
			array( 'default' => '#000' ),
			$render_slug
		);

		// Custom background.
		$this->get_custom_bg_style( $render_slug, 'overlay', '%%order_class%% .wdc-insta-overlay', '' );

		$this->get_responsive_styles(
			'insta_icon_size',
			'%%order_class%% .wdc-insta-icon svg',
			array( 'primary' => 'height' ),
			array( 'default' => '32px' ),
			$render_slug
		);

		$this->get_responsive_styles(
			'insta_icon_size',
			'%%order_class%% .wdc-insta-icon svg',
			array( 'primary' => 'width' ),
			array( 'default' => '32px' ),
			$render_slug
		);

		if ( 'auto' !== $this->props['image_height'] ) {
			$this->get_responsive_styles(
				'image_height',
				'%%order_class%% .wdc-insta-media-container',
				array( 'primary' => 'height' ),
				array( 'default' => 'auto' ),
				$render_slug
			);
		}

	}

}

new WDC_InstagramFeed();
