<?php
namespace WowDiviCarousel\Api;

class Google_My_Business {

    const CACHE_DURATION = 86400;
    const ERROR_DURATION = 86400;

    public $base    = '';
    public $api_url = false;
    private $api_key;
    private $site_key;
    private $token;

    public function __construct() {
        $this->api_key  = get_option( 'wdc_api_key' );
        $this->site_key = get_option( 'wdc_site_key' );
        $this->token    = get_option( 'wdc_api_token' );
    }

    public function init() {
        $this->api_url = apply_filters( 'wdc_api_url', trailingslashit( $this->base ) );
        add_action( 'admin_post_wdc_generate_url', array( $this, 'generate_url' ) );
        add_action( 'admin_post_wdc_disconnect', array( $this, 'unlink_site' ) );
        add_action( 'admin_post_wdc_response', array( $this, 'handle_response' ) );
        add_action( 'wdc_refresh_token', array( $this, 'refresh_token' ) );
        $this->check_refresh_event();
    }

    public function check_refresh_event() {
        if ( !wp_next_scheduled( 'wdc_refresh_token' ) ) {
            wp_schedule_event( time(), 'daily', 'wdc_refresh_token' );
        }
    }

    public function generate_url() {

        if ( !current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'No permission to edit options', '' ) );
        }

        $ip = '';
        if ( !empty( $_SERVER['HTTP_CLIENT_IP'] ) && !filter_var( $_SERVER['HTTP_CLIENT_IP'], FILTER_VALIDATE_IP ) === false ) {
            $ip = $_SERVER['HTTP_CLIENT_IP']; //phpcs:ignore
        } elseif ( !empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) && !filter_var( $_SERVER['HTTP_X_FORWARDED_FOR'], FILTER_VALIDATE_IP ) === false ) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR']; //phpcs:ignore
        } else {
            $ip = $_SERVER['REMOTE_ADDR']; //phpcs:ignore
        }

        $response = wp_remote_post(
            $this->api_url . 'google/authlink/',
            array(
                'timeout' => 20,
                'body'    => array(
                    'post_url' => esc_url( admin_url( 'admin-post.php' ) ),
                    'nonce'    => wp_create_nonce( 'wdc_generate_url' ),
                    'uip'      => $ip,
                    'dp'       => $_SERVER['REQUEST_URI'], //phpcs:ignore
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            wp_die( sprintf( esc_html__( 'Something went wrong: %s', '' ), $error_message ) ); //phpcs:ignore
        }

        $data = json_decode( $response['body'] );

        if ( !empty( $data->key ) ) {
            update_option( 'wdc_request_key', sanitize_key( $data->key ) );
        }
        delete_transient( 'wdc_get_accounts_cache' );
        wp_safe_redirect( $data->url );
        exit;
    }

    public function handle_response() {

        if ( !current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'No permission to edit options', '' ) );
        }

        if ( !wp_verify_nonce( sanitize_key( $_REQUEST['nonce'] ), 'wdc_generate_url' ) ) { //phpcs:ignore
            wp_die( esc_html__( 'Invalid nonce', '' ) );
        }

        update_option( 'wdc_site_key', sanitize_key( $_REQUEST['sitekey'] ) ); //phpcs:ignore
        update_option( 'wdc_api_key', sanitize_key( $_REQUEST['key'] ) ); //phpcs:ignore
        update_option( 'wdc_api_token', sanitize_key( $_REQUEST['token'] ) ); //phpcs:ignore

        wp_safe_redirect( admin_url( 'admin.php?page=wow-divi-carousel#integration' ) );
        exit;
    }

    protected function do_get( $endpoint, $args = array() ) {

        $args['timeout'] = 20;
        $args['apiKey']  = $this->site_key;
        $args['token']   = $this->token;

        $url      = add_query_arg( $args, $this->api_url . $endpoint );
        $response = wp_remote_get( $url );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            set_transient( 'wdc_api_error', $error_message, self::ERROR_DURATION );
            return $error_message;

        } else {
            $data = json_decode( wp_remote_retrieve_body( $response ) );
            if ( !isset( $data->error ) ) {
                return $data;
            } else {
                return $data->error;
            }
        }
    }

    public function get_accounts() {
        if ( !wdc_is_configured() ) {
            return;
        }
        $cached = get_transient( 'wdc_get_accounts_cache' );
        if ( false === $cached ) {

            $accounts = $this->do_get( 'google/get_accounts/' );
            set_transient( 'wdc_get_accounts_cache', $accounts, self::CACHE_DURATION );
            return $accounts;
        }
        return $cached;
    }

    public function get_locations( $account_name ) {

        if ( !$account_name ) {
            return false;
        }

        $transient_name = 'wdc_get_locations_cache_' . str_replace( '/', '_', $account_name );
        $cached = get_transient( $transient_name );
        if ( false === $cached ) {
            $args['account_name'] = $account_name;
            $locations = $this->do_get( 'google/get_locations/', $args );
            set_transient( $transient_name, $locations, self::CACHE_DURATION );
            return $locations;
        }
        return (object) $cached;
    }

    public function get_reviews( $location_name ) {

        if ( !$location_name ) {
            return false;
        }

        $args['location_name'] = $location_name;
        $reviews               = $this->do_get( 'google/get_reviews/', $args );
        return $reviews;
    }

    public function refresh_token() {

        if ( !wdc_is_configured() ) {
            return;
        }

        $result = $this->do_get( 'google/refreshtoken/' );
        $token  = $result->token;
        if ( !$token ) {
            return false;
        }

        update_option( 'wdc_api_token', sanitize_key( $token ) );
        return true;
    }

    public function unlink_site() {

        update_option( 'wdc_api_key', false );
        update_option( 'wdc_site_key', false );
        update_option( 'wdc_api_token', false );
        delete_transient( 'wdc_get_accounts_cache' );

        // Redirect.
        wp_safe_redirect( admin_url( 'admin.php?page=wow-divi-carousel#integration' ) );
    }
}
